import {
    Component,
    ViewEncapsulation,
    ChangeDetectionStrategy,
    HostBinding,
    OnInit,
    ElementRef, OnDestroy
} from '@angular/core';
import { Select, Store } from '@ngxs/store';
import { DriveState } from '../../state/drive-state';
import { DriveEntry } from '../../files/models/drive-entry';
import { Observable, Subscription } from 'rxjs';
import { DrivePageType } from '../../state/models/available-pages';
import { Keybinds } from 'common/core/keybinds/keybinds.service';
import { DeleteSelectedEntries, SelectAllEntries } from '../../state/actions/commands';
import { EntryDoubleTapped } from '../../state/actions/events';

@Component({
    selector: 'entries-container',
    templateUrl: './entries-container.component.html',
    styleUrls: ['./entries-container.component.scss'],
    encapsulation: ViewEncapsulation.None,
    changeDetection: ChangeDetectionStrategy.OnPush,
    providers: [Keybinds],
})
export class EntriesContainerComponent implements OnInit, OnDestroy {
    @HostBinding('attr.tabindex') tabindex = 0;
    @Select(DriveState.entries) entries$: Observable<DriveEntry[]>;
    @Select(DriveState.entriesEmpty) noEntries: Observable<boolean>;
    @Select(DriveState.viewMode) viewMode$: Observable<'list'|'grid'>;
    @Select(DriveState.activePageName) activePageName$: Observable<DrivePageType>;
    private keybindSub: Subscription;

    constructor(
        private store: Store,
        private el: ElementRef,
        private keybinds: Keybinds,
    ) {}

    ngOnInit() {
        this.keybindSub = this.keybinds.listenOn(this.el.nativeElement);
        this.keybinds.addWithPreventDefault('ctrl+a', () => {
            this.store.dispatch(new SelectAllEntries());
        });
        this.keybinds.addWithPreventDefault('delete', () => {
            this.store.dispatch(new DeleteSelectedEntries());
        });
        this.keybinds.addWithPreventDefault('enter', () => {
            this.store.dispatch(new EntryDoubleTapped(this.store.selectSnapshot(DriveState.selectedEntry)));
        });
    }

    ngOnDestroy() {
        this.keybindSub.unsubscribe();
    }
}
